/*
 * "Crappy Sprite Font", data generator
 * (C) Copyright 2019 ccr/TNSP^PWP <ccr@tnsp.org>
 *
 * This code is licensed under WTFPL (see COPYING.txt)
 *
 * This code generates the data used by "fancy.asm".
 */
#include <stdio.h>
#include <string.h>
#include <stdint.h>


#define SET_SPR_WIDTH_PX  24
#define SET_SPR_HEIGHT_PX 21

#define SET_WIDTH_SEGS 3
#define SET_HEIGHT_SEGS 3

#define SET_SEG_WIDTH_PX 8
#define SET_SEG_HEIGHT_PX 7


enum
{
    MFLIP_X = 0x08,
    MFLIP_Y = 0x10,
    MMASK   = 0x07,
};

typedef struct
{
    uint8_t segs[SET_WIDTH_SEGS * SET_HEIGHT_SEGS];
} DMChar;


typedef struct
{
    uint8_t data[SET_SPR_WIDTH_PX * SET_SPR_HEIGHT_PX];
} DMSprite;


typedef struct
{
    uint8_t data[SET_SEG_WIDTH_PX * SET_SEG_HEIGHT_PX];
} DMSeg;



static const DMSeg dmSegs[] =
{
    {{
        "........"
        "........"
        "........"
        "........"
        "........"
        "........"
        "........"
    }},
    {{
        "......##"
        "...#####"
        ".#######"
        "########"
        "######.."
        "######.."
        "######.."
    }},
    {{
        "########"
        "########"
        "########"
        "........"
        "........"
        "........"
        "........"
    }},
    {{
        "######.."
        "######.."
        "######.."
        "######.."
        "######.."
        "######.."
        "######.."
    }},
    {{
        "##....##"
        "###..###"
        "########"
        "########"
        ".######."
        "..####.."
        "...##..."
    }},
    {{
        "########"
        "########"
        "########"
        "######.."
        "######.."
        "######.."
        "######.."
    }},
    {{
        "..######"
        ".######."
        "######.."
        "#####..."
        "######.."
        ".######."
        "..######"
    }},
    {{
        "....####"
        "...#####"
        "..######"
        ".######."
        "######.."
        "#####..."
        "####...."
    }},
};

static const int ndmSegs = sizeof(dmSegs) / sizeof(dmSegs[0]);


typedef struct _DMBitStreamContext
{
    void *handle;

    int (*putByte)(struct _DMBitStreamContext *ctx);
    int (*getByte)(struct _DMBitStreamContext *ctx);

    unsigned int
        outBuf, outBitCount, outByteCount,
        inBuf, inBitCount, inByteCount;
} DMBitStreamContext;


void dmInitBitStreamContext(DMBitStreamContext *ctx)
{
    ctx->outBuf       = 0;
    ctx->outByteCount = 0;
    ctx->outBitCount  = 0;

    ctx->inBuf        = 0;
    ctx->inByteCount  = 0;
    ctx->inBitCount   = 0;
}


int dmPutBits(DMBitStreamContext *ctx, const unsigned int val, const unsigned int n)
{
    for (unsigned int i = 0; i < n; i++)
    {
        ctx->outBuf <<= 1;

        if (val & (1 << (n - 1 - i)))
            ctx->outBuf |= 1;

        ctx->outBitCount++;
        if (ctx->outBitCount == 8)
        {
            int ret;
            if ((ret = ctx->putByte(ctx)) != 0)
                return ret;

            ctx->outBitCount = 0;
            ctx->outByteCount++;
        }
    }

    return 0;
}


int dmGetBits(DMBitStreamContext *ctx, unsigned int *val, const unsigned int n)
{
    *val = 0;

    for (unsigned int i = 0; i < n; i++)
    {
        if (ctx->inBitCount == 0)
        {
            int ret;
            if ((ret = ctx->getByte(ctx)) != 0)
                return ret;

            ctx->inBitCount = 8;
            ctx->inByteCount++;
        }

        *val <<= 1;
        *val |= ctx->inBuf >> 7 & 1;

        ctx->inBuf <<= 1;
        ctx->inBitCount--;
    }

    return 0;
}


int dmFlushBitStream(DMBitStreamContext *ctx)
{
  if (ctx == NULL)
      return -2;

  if (ctx->outBitCount > 0)
      return dmPutBits(ctx, 0, 8 - ctx->outBitCount);

  return 0;
}


int dmPutByteFILE(DMBitStreamContext *ctx)
{
    return fputc(ctx->outBuf & 0xff, (FILE *) ctx->handle) != EOF ? 0 : -1;
}


void dmPrintSprite(FILE *fp, const DMSprite *spr)
{
    for (int yc = 0; yc < SET_SPR_HEIGHT_PX; yc++)
    {
        const uint8_t *dp = spr->data + yc * SET_SPR_WIDTH_PX;
        for (int xc = 0; xc < SET_SPR_WIDTH_PX; xc++)
        {
            fputc(dp[xc] ? '#' : '.', fp);
        }
        fputs("\n", fp);
    }
}


void dmDrawChar(DMSprite *spr, const DMChar *chr)
{
    for (int sy = 0; sy < SET_WIDTH_SEGS; sy++)
    for (int sx = 0; sx < SET_HEIGHT_SEGS; sx++)
    {
        uint8_t seg = chr->segs[sy * SET_WIDTH_SEGS + sx];
        const uint8_t *spp = dmSegs[seg & MMASK].data;
        uint8_t *dpp = spr->data + SET_SPR_WIDTH_PX * SET_SEG_HEIGHT_PX * sy + SET_SEG_WIDTH_PX * sx;

        for (int yc = 0; yc < SET_SEG_HEIGHT_PX; yc++)
        {
            const uint8_t *sp = spp + ((seg & MFLIP_Y) ? (SET_SEG_HEIGHT_PX - yc - 1) : yc) * SET_SEG_WIDTH_PX;
            uint8_t *dp = dpp + yc * SET_SPR_WIDTH_PX;

            for (int xc = 0; xc < SET_SEG_WIDTH_PX; xc++)
                dp[xc] = sp[(seg & MFLIP_X) ? (SET_SEG_WIDTH_PX - xc - 1) : xc] == '.' ? 0 : 1;
        }
    }
}


static const DMChar dmCharacters[] =
{
    {{ // A
        1, 2, 1 | MFLIP_X,
        5, 2, 5 | MFLIP_X,
        3, 0, 3 | MFLIP_X,
    }},
    {{ // B
        5, 2, 1 | MFLIP_X,
        3, 2, 6,
        5 | MFLIP_Y, 2 | MFLIP_Y, 1 | MFLIP_X | MFLIP_Y,
    }},
    {{ // C
        1, 2, 1 | MFLIP_X,
        3, 0, 0,
        1 | MFLIP_Y, 2 | MFLIP_Y, 1 | MFLIP_X | MFLIP_Y
    }},
    {{ // D
        5, 2, 1 | MFLIP_X,
        3, 0, 3 | MFLIP_X,
        5 | MFLIP_Y, 2 | MFLIP_Y, 1 | MFLIP_X | MFLIP_Y
    }},
    {{ // E
        5, 2, 2,
        5, 2, 0,
        5 | MFLIP_Y, 2 | MFLIP_Y, 2 | MFLIP_Y,
    }},
    {{ // F
        5, 2, 2,
        5, 2, 0,
        3, 0, 0,
    }},
    {{ // G
        1, 2, 1 | MFLIP_X,
        3, 2 | MFLIP_Y, 2 | MFLIP_Y,
        1 | MFLIP_Y, 2 | MFLIP_Y, 1 | MFLIP_X | MFLIP_Y
    }},
    {{ // H
        3, 0, 3 | MFLIP_X,
        5, 2, 5 | MFLIP_X,
        3, 0, 3 | MFLIP_X,
    }},
    {{ // I
        0, 3, 0,
        0, 3, 0,
        0, 3, 0,
    }},
    {{ // J
        0, 0, 3 | MFLIP_X,
        0, 0, 3 | MFLIP_X,
        1 | MFLIP_Y, 2 | MFLIP_Y, 1 | MFLIP_X | MFLIP_Y,
    }},
    {{ // K
        3, 0, 7,
        3, 7, 0,
        3, 0, 7 | MFLIP_Y,
    }},
    {{ // L
        3, 0, 0,
        3, 0, 0,
        5 | MFLIP_Y, 2 | MFLIP_Y, 2 | MFLIP_Y,
    }},
    {{ // M
        5, 4, 5 | MFLIP_X,
        3, 0, 3 | MFLIP_X,
        3, 0, 3 | MFLIP_X,
    }},
    {{ // N
        3, 0, 3 | MFLIP_X,
        3, 7 | MFLIP_Y, 3 | MFLIP_X,
        3, 0, 3 | MFLIP_X,
    }},
    {{ // O
        1, 2, 1 | MFLIP_X,
        3, 0, 3 | MFLIP_X,
        1 | MFLIP_Y, 2 | MFLIP_Y, 1 | MFLIP_X | MFLIP_Y
    }},
    {{ // P
        5, 2, 1 | MFLIP_X,
        5 | MFLIP_Y, 2 | MFLIP_Y, 1 | MFLIP_X | MFLIP_Y,
        3, 0, 0,
    }},
    {{ // Q
        1, 2, 1 | MFLIP_X,
        3, 0, 3 | MFLIP_X,
        1 | MFLIP_Y, 2 | MFLIP_Y, 7 | MFLIP_Y
    }},
    {{ // R
        5, 2, 1 | MFLIP_X,
        5 | MFLIP_Y, 2 | MFLIP_Y, 1 | MFLIP_X | MFLIP_Y,
        3, 0, 7 | MFLIP_Y,
    }},
    {{ // S
        1, 2, 2,
        2, 2, 1 | MFLIP_X,
        2 | MFLIP_Y, 2 | MFLIP_Y, 1 | MFLIP_X | MFLIP_Y
    }},
    {{ // T
        2, 3, 2,
        0, 3, 0,
        0, 3, 0,
    }},
    {{ // U
        3, 0, 3 | MFLIP_X,
        3, 0, 3 | MFLIP_X,
        1 | MFLIP_Y, 2 | MFLIP_Y, 1 | MFLIP_X | MFLIP_Y
    }},
    {{ // V
        3, 0, 3 | MFLIP_X,
        7 | MFLIP_X, 0, 7,
        0, 4, 0,
    }},
    {{ // W
        3, 0, 3 | MFLIP_X,
        3, 0, 3 | MFLIP_X,
        5 | MFLIP_Y, 4 | MFLIP_Y, 5 | MFLIP_X | MFLIP_Y,
    }},
    {{ // X
        7 | MFLIP_X, 0, 7,
        0, 7, 0,
        7, 0, 7 | MFLIP_X,
    }},
    {{ // Y
        7 | MFLIP_X, 0, 7,
        0, 4, 0,
        0, 3, 0,
    }},
    {{ // Z
        2, 2, 7,
        0, 7, 0,
        7, 2 | MFLIP_Y, 2 | MFLIP_Y,
    }},
    {{ // 0
        1, 2, 1 | MFLIP_X,
        3, 7, 3 | MFLIP_X,
        1 | MFLIP_Y, 2 | MFLIP_Y, 1 | MFLIP_X | MFLIP_Y
    }},
    {{ // 1
        7, 5 | MFLIP_X, 0,
        0, 3 | MFLIP_X, 0,
        0, 3 | MFLIP_X, 0,
    }},
    {{ // 2
        2, 2, 1 | MFLIP_X,
        1, 2, 2,
        5 | MFLIP_Y, 2 | MFLIP_Y, 2 | MFLIP_Y,
    }},
    {{ // 3
        2, 2, 1 | MFLIP_X,
        0, 2, 5 | MFLIP_X,
        2 | MFLIP_Y, 2 | MFLIP_Y, 1 | MFLIP_X | MFLIP_Y
    }},
    {{ // 4
        7, 0, 3 | MFLIP_X,
        1 | MFLIP_Y, 2 | MFLIP_Y, 5 | MFLIP_X | MFLIP_Y,
        0, 0, 3 | MFLIP_X,
    }},
    {{ // 5
        5, 2, 2,
        2, 2, 1 | MFLIP_X,
        2 | MFLIP_Y, 2 | MFLIP_Y, 1 | MFLIP_X | MFLIP_Y
    }},
    {{ // 6
        1, 2, 0,
        5, 2, 1 | MFLIP_X,
        1 | MFLIP_Y, 2 | MFLIP_Y, 1 | MFLIP_X | MFLIP_Y
    }},
    {{ // 7
        2, 2, 1 | MFLIP_X,
        0, 2, 5 | MFLIP_X,
        0, 0, 3 | MFLIP_X,
    }},
    {{ // 8
        1, 2, 1 | MFLIP_X,
        6 | MFLIP_X, 2, 6,
        1 | MFLIP_Y, 2 | MFLIP_Y, 1 | MFLIP_X | MFLIP_Y
    }},
    {{ // 9
        1, 2, 1 | MFLIP_X,
        1 | MFLIP_Y, 2 | MFLIP_Y, 5 | MFLIP_X | MFLIP_Y,
        0, 2 | MFLIP_Y, 1 | MFLIP_X | MFLIP_Y,
    }},
};


static const int ndmCharacters = sizeof(dmCharacters) / sizeof(dmCharacters[0]);


int main(int argc, char *argv[])
{
    DMBitStreamContext ctx;
    FILE *outFile = stdout, *dbgFile = stderr;
    char *mode, *prog = argv[0];

    if (argc < 2)
    {
        fprintf(stderr, "Usage: %s <ascii|segs|defs>\n",
            prog);
        return 0;
    }
    mode = argv[1];

    dmInitBitStreamContext(&ctx);
    ctx.handle = (void *) outFile;
    ctx.putByte = dmPutByteFILE;

    if (mode[0] == 'a')
    {
        for (int i = 0; i < ndmCharacters; i++)
        {
            DMSprite spr;
            dmDrawChar(&spr, &dmCharacters[i]);
            fprintf(outFile, "\n");
            dmPrintSprite(outFile, &spr);
        }
    }
    else
    if (mode[0] == 's')
    {
        for (int i = 0; i < ndmSegs; i++)
        {
            const DMSeg *seg = &dmSegs[i];

            for (int yc = 0; yc < SET_SEG_HEIGHT_PX; yc++)
            {
                const uint8_t *sp = seg->data + yc * SET_SEG_WIDTH_PX;

                for (int xc = 0; xc < SET_SEG_WIDTH_PX; xc++)
                {
                     dmPutBits(&ctx, sp[xc] == '.' ? 0 : 1, 1);
                }
            }
            //dmPutBits(&ctx, 0xaa, 8);
        }
        dmFlushBitStream(&ctx);
    }
    else
    if (mode[0] == 'd')
    {
        for (int i = 0; i < ndmCharacters; i++)
        {
            const DMChar *chr = &dmCharacters[i];
            for (int sy = 0; sy < SET_HEIGHT_SEGS; sy++)
            for (int sx = 0; sx < SET_WIDTH_SEGS; sx++)
            {
                uint8_t seg = chr->segs[sy * SET_WIDTH_SEGS + sx];
                dmPutBits(&ctx, seg, 5);
            }
        }
        dmFlushBitStream(&ctx);
    }

    return 0;
}
